<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ErrorLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;

class ErrorLogController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'can:view error logs']);
    }

    /**
     * Display a listing of error logs
     */
    public function index(Request $request)
    {
        $query = ErrorLog::with('user')->latest();

        // Filter by level
        if ($request->filled('level')) {
            $query->where('level', $request->level);
        }

        // Filter by resolved status
        if ($request->filled('status')) {
            if ($request->status === 'resolved') {
                $query->resolved();
            } elseif ($request->status === 'unresolved') {
                $query->unresolved();
            }
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('message', 'like', "%{$search}%")
                  ->orWhere('type', 'like', "%{$search}%")
                  ->orWhere('file', 'like', "%{$search}%");
            });
        }

        // Date range filter
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $errors = $query->paginate(20);

        $stats = [
            'total' => ErrorLog::count(),
            'unresolved' => ErrorLog::unresolved()->count(),
            'critical' => ErrorLog::critical()->count(),
            'today' => ErrorLog::whereDate('created_at', today())->count(),
        ];

        return view('admin.errors.index', compact('errors', 'stats'));
    }

    /**
     * Display the specified error log
     */
    public function show(ErrorLog $error)
    {
        $error->load('user');
        
        // Similar errors
        $similarErrors = ErrorLog::where('type', $error->type)
            ->where('id', '!=', $error->id)
            ->latest()
            ->take(5)
            ->get();

        return view('admin.errors.show', compact('error', 'similarErrors'));
    }

    /**
     * Mark error as resolved
     */
    public function resolve(ErrorLog $error)
    {
        $error->markAsResolved();

        return response()->json([
            'success' => true,
            'message' => 'Error marked as resolved',
        ]);
    }

    /**
     * Mark multiple errors as resolved
     */
    public function bulkResolve(Request $request)
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:error_logs,id',
        ]);

        ErrorLog::whereIn('id', $request->ids)->update(['is_resolved' => true]);

        return response()->json([
            'success' => true,
            'message' => count($request->ids) . ' error(s) marked as resolved',
        ]);
    }

    /**
     * Delete an error log
     */
    public function destroy(ErrorLog $error)
    {
        $error->delete();

        return response()->json([
            'success' => true,
            'message' => 'Error log deleted successfully',
        ]);
    }

    /**
     * Delete multiple error logs
     */
    public function bulkDelete(Request $request)
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:error_logs,id',
        ]);

        ErrorLog::whereIn('id', $request->ids)->delete();

        return response()->json([
            'success' => true,
            'message' => count($request->ids) . ' error(s) deleted successfully',
        ]);
    }

    /**
     * Download error log as JSON
     */
    public function download(ErrorLog $error)
    {
        $data = [
            'id' => $error->id,
            'level' => $error->level,
            'type' => $error->type,
            'message' => $error->message,
            'file' => $error->file,
            'line' => $error->line,
            'trace' => $error->trace,
            'url' => $error->url,
            'method' => $error->method,
            'ip' => $error->ip,
            'user_agent' => $error->user_agent,
            'user' => $error->user ? [
                'id' => $error->user->id,
                'name' => $error->user->name,
                'email' => $error->user->email,
            ] : null,
            'context' => $error->context,
            'created_at' => $error->created_at->toIso8601String(),
        ];

        $filename = 'error-' . $error->id . '-' . date('Y-m-d-His') . '.json';

        return response()->json($data)
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"')
            ->header('Content-Type', 'application/json');
    }

    /**
     * Download multiple errors as JSON
     */
    public function bulkDownload(Request $request)
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:error_logs,id',
        ]);

        $errors = ErrorLog::with('user')->whereIn('id', $request->ids)->get();

        $data = $errors->map(function($error) {
            return [
                'id' => $error->id,
                'level' => $error->level,
                'type' => $error->type,
                'message' => $error->message,
                'file' => $error->file,
                'line' => $error->line,
                'trace' => $error->trace,
                'url' => $error->url,
                'method' => $error->method,
                'ip' => $error->ip,
                'user_agent' => $error->user_agent,
                'user' => $error->user ? [
                    'id' => $error->user->id,
                    'name' => $error->user->name,
                    'email' => $error->user->email,
                ] : null,
                'context' => $error->context,
                'created_at' => $error->created_at->toIso8601String(),
            ];
        });

        $filename = 'errors-bulk-' . date('Y-m-d-His') . '.json';

        return response()->json($data)
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"')
            ->header('Content-Type', 'application/json');
    }

    /**
     * Send error to support dashboard
     */
    public function sendToSupport(ErrorLog $error)
    {
        try {
            $response = Http::timeout(10)->post('https://dash.siedestudios.co.uk/api/error-reports', [
                'license_key' => config('app.license_key') ?? env('LICENSE_KEY') ?? 'no-key-set',
                'domain' => request()->getHost(),
                'error' => [
                    'level' => $error->level,
                    'type' => $error->type,
                    'message' => $error->message,
                    'file' => $error->file,
                    'line' => $error->line,
                    'trace' => $error->trace,
                    'url' => $error->url,
                    'method' => $error->method,
                    'context' => $error->context,
                    'occurred_at' => $error->created_at->toIso8601String(),
                ],
                'system_info' => [
                    'php_version' => PHP_VERSION,
                    'laravel_version' => app()->version(),
                    'server' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
                ],
            ]);

            if ($response->successful()) {
                $error->markAsSent();

                return response()->json([
                    'success' => true,
                    'message' => 'Error report sent successfully to support dashboard',
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'Failed to send error report: ' . $response->body(),
            ], 500);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to send error report: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Send multiple errors to support dashboard
     */
    public function bulkSendToSupport(Request $request)
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:error_logs,id',
        ]);

        $errors = ErrorLog::whereIn('id', $request->ids)->get();
        $successCount = 0;
        $failCount = 0;

        foreach ($errors as $error) {
            try {
                $response = Http::timeout(10)->post('https://dash.siedestudios.co.uk/api/error-reports', [
                    'license_key' => config('app.license_key') ?? env('LICENSE_KEY') ?? 'no-key-set',
                    'domain' => request()->getHost(),
                    'error' => [
                        'level' => $error->level,
                        'type' => $error->type,
                        'message' => $error->message,
                        'file' => $error->file,
                        'line' => $error->line,
                        'trace' => $error->trace,
                        'url' => $error->url,
                        'method' => $error->method,
                        'context' => $error->context,
                        'occurred_at' => $error->created_at->toIso8601String(),
                    ],
                    'system_info' => [
                        'php_version' => PHP_VERSION,
                        'laravel_version' => app()->version(),
                        'server' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
                    ],
                ]);

                if ($response->successful()) {
                    $error->markAsSent();
                    $successCount++;
                } else {
                    $failCount++;
                }
            } catch (\Exception $e) {
                $failCount++;
            }
        }

        return response()->json([
            'success' => $successCount > 0,
            'message' => "{$successCount} error(s) sent successfully" . ($failCount > 0 ? ", {$failCount} failed" : ""),
        ]);
    }

    /**
     * Clear all resolved errors
     */
    public function clearResolved()
    {
        $count = ErrorLog::resolved()->count();
        ErrorLog::resolved()->delete();

        return response()->json([
            'success' => true,
            'message' => "{$count} resolved error(s) cleared successfully",
        ]);
    }

    /**
     * Clear old errors (older than 30 days)
     */
    public function clearOld()
    {
        $count = ErrorLog::where('created_at', '<', now()->subDays(30))->count();
        ErrorLog::where('created_at', '<', now()->subDays(30))->delete();

        return response()->json([
            'success' => true,
            'message' => "{$count} old error(s) cleared successfully",
        ]);
    }
}